/*
 * Stems, an Experiment in Self Organizing Construction
 * Competition Entry for AI junkie October 2003
 * http://www.ai-junkie.com/
 *
 * Copyright (C) David Lotts 2003
 * This code is distributed under the GNU General Public License
 *    Go here for details: http://www.gnu.org/copyleft/gpl.html
 *    Email the author at dlotts@pobox.com
 *
 * Credits:
 * The OpenGL setup code came from Jeff Molofee '99 
 *     Visit Jeff at http://nehe.gamedev.net/
 * ported to Linux/SDL by Ti Leggett '01
 *     email to leggett@eecs.tulane.edu
 *
 * Great resource for Geometry and Trig: 
 *       http://mathworld.wolfram.com
 *
 * Concise C++ reference:
 *      http://www.cppreference.com/
 * 
 * 
 */
// this is the highth and width of a stem.  Many things use this.
float const static stemHi = 0.4f;
float const static stemWi = 0.08f;
 
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#ifdef win32
    #include <windows.h>
#endif
#include<iostream> 
#include<vector>
//#include<algorithm>
#include <OpenGL/gl.h>
#include <OpenGL/glext.h>
#include <OpenGL/glu.h>
#include <math.h>
#ifndef M_PI  // needed on macos
#define M_PI pi  
#endif
#ifdef linux
   #include "SDL/SDL.h"
#else
   #include "SDL.h"
#endif
#include "OSUtil.h"
using namespace std;

/* screen width, height, and bit depth */
#define SCREEN_WIDTH  800 //640
#define SCREEN_HEIGHT 600 //480
#define SCREEN_BPP     32//16

/* Set up some booleans */
#define TRUE  1
#define FALSE 0

// called before defined.
int drawGLScene( GLvoid );
// draw a stem.
void drawStem();
/* This is our SDL surface */
SDL_Surface *surface;

    // current camera
    static float cameraMatrix[16];

int light = TRUE; /* Whether or not lighting is on */
int blend = FALSE; /* Whether or not blending is on */

/* Ambient Light Values  */
GLfloat LightAmbient[]  = { 0.7f, 0.7f, 0.7f, 1.0f };
/* Diffuse Light Values */
GLfloat LightDiffuse[]  = { 0.3f, 0.3f, 0.3f, 1.0f };
/* Light Position */
GLfloat LightPosition[] = {1.0f, 20.0f, 1.0f, 0.0f };


GLuint filter=2;     /* Which Filter To Use */
//GLuint texture1[3]; /* Storage for 3 textures */
GLuint texture2[3]; /* Storage for 3 textures */

bool zoomIn=FALSE, 
    zoomOut=FALSE, 
    rotUp=FALSE, 
    rotDown=FALSE, 
    rotLeft=FALSE, 
    rotRight=FALSE;
    
/* function to release/destroy our resources and restoring the old desktop */
void Quit( int returnCode )
{
    /* clean up the window */
    SDL_Quit( );

    /* and exit appropriately */
    exit( returnCode );
}

/* function to load in bitmap as a GL texture */
int LoadGLTextures(GLuint texture[], char * myBmp)
{
    /* Status indicator */
    int Status = FALSE;

    /* Create storage space for the texture */
    SDL_Surface *TextureImage[1]; 

    /* Load The Bitmap, Check For Errors, If Bitmap's Not Found Quit */
    if ( ( TextureImage[0] = SDL_LoadBMP( myBmp) ) )
        {

	    /* Set the status to true */
	    Status = TRUE;

	    /* Create The Texture */
	    glGenTextures( 3, &texture[0] );

	    /* Load in texture 1 */
	    /* Typical Texture Generation Using Data From The Bitmap */
	    glBindTexture( GL_TEXTURE_2D, texture[0] );

	    /* Generate The Texture */
	    glTexImage2D( GL_TEXTURE_2D, 0, 3, TextureImage[0]->w,
			  TextureImage[0]->h, 0, GL_BGR,
			  GL_UNSIGNED_BYTE, TextureImage[0]->pixels );
	    
	    /* Nearest Filtering */
	    glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER,
			     GL_NEAREST );
	    glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER,
			     GL_NEAREST );

	    /* Load in texture 2 */
	    /* Typical Texture Generation Using Data From The Bitmap */
	    glBindTexture( GL_TEXTURE_2D, texture[1] );

	    /* Linear Filtering */
	    glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER,
			     GL_LINEAR );
	    glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER,
			     GL_LINEAR );

	    /* Generate The Texture */
	    glTexImage2D( GL_TEXTURE_2D, 0, 3, TextureImage[0]->w,
			  TextureImage[0]->h, 0, GL_BGR,
			  GL_UNSIGNED_BYTE, TextureImage[0]->pixels );

	    /* Load in texture 3 */
	    /* Typical Texture Generation Using Data From The Bitmap */
	    glBindTexture( GL_TEXTURE_2D, texture[2] );

	    /* Mipmapped Filtering */
	    glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER,
			     GL_LINEAR_MIPMAP_NEAREST );
	    glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER,
			     GL_LINEAR );

	    /* Generate The MipMapped Texture */
	    gluBuild2DMipmaps( GL_TEXTURE_2D, 3, TextureImage[0]->w,
			       TextureImage[0]->h, GL_BGR,
			       GL_UNSIGNED_BYTE, TextureImage[0]->pixels );
        }

    /* Free up any memory we may have used */
    if ( TextureImage[0] )
	    SDL_FreeSurface( TextureImage[0] );

    return Status;
}

/* function to reset our viewport after a window resize */
int resizeWindow( int width, int height )
{
    /* Height / width ration */
    GLfloat ratio;
 
    /* Protect against a divide by zero */
    if ( height == 0 )
	height = 1;

    ratio = ( GLfloat )width / ( GLfloat )height;

    /* Setup our viewport. */
    glViewport( 0, 0, ( GLint )width, ( GLint )height );

    /* change to the projection cameraMatrix and set our viewing volume. */
    glMatrixMode( GL_PROJECTION );
    glLoadIdentity( );

    /* Set our perspective */
    gluPerspective( 45.0f, ratio, 0.1f, 100.0f );

    /* Make sure we're chaning the model view and not the projection */
    glMatrixMode( GL_MODELVIEW );

    /* Reset The View */
    glLoadIdentity( );

    return( TRUE );
}

/* function to handle key press events */
void handleKeyPress( SDL_keysym *keysym, bool isKeydown )
{
    switch ( keysym->sym )
	{
	case SDLK_ESCAPE:
	    /* ESC key was pressed */
	    Quit( 0 );
	    break;
	case SDLK_b:
	    /* 'b' key was pressed
	     * this toggles blending
	     */
	    if (isKeydown) 
	    {
    	    blend = !blend;
    	    if ( blend )
    		{
    		    glEnable( GL_BLEND );
    		    glDisable( GL_DEPTH_TEST );
    		}
    	    else
    		{
    		    glDisable( GL_BLEND );
    		    glEnable( GL_DEPTH_TEST );
    		}
		}
	    break;
	case SDLK_f:
	    /* 'f' key was pressed
	     * this pages through the different filters
	     */
	    if (isKeydown) 
    	    filter = ( ++filter ) % 3;
	    break;
	case SDLK_l:
	    /* 'l' key was pressed
	     * this toggles the light
	     */
	    if (isKeydown) 
	    {
            light = !light;
    	    if ( !light )
        		glDisable( GL_LIGHTING );
    	    else
        		glEnable( GL_LIGHTING );
		}
	    break;
	case SDLK_PAGEUP:
	    /* PageUp key was pressed
	     * this zooms into the scene
	     */
   	    zoomIn = isKeydown;
        //    z -= 0.02f;
	    break;
	case SDLK_PAGEDOWN:
	    /* PageDown key was pressed
	     * this zooms out of the scene
	     */
   	    zoomOut = isKeydown;
	    // z += 0.02f;
	    break;
	case SDLK_UP:
	    /* Up arrow key was pressed
	     * this affects the x rotation
	     */
   	    rotUp = isKeydown;
	    //xspeed -= 0.01f;
	    break;
	case SDLK_DOWN:
	    /* Down arrow key was pressed
	     * this affects the x rotation
	     */
   	    rotDown = isKeydown;
	    //xspeed += 0.01f;
	    break;
	case SDLK_RIGHT:
	    /* Right arrow key was pressed
	     * this affects the y rotation
	     */
   	    rotRight = isKeydown;
	    //yspeed += 0.01f;
	    break;
	case SDLK_LEFT:
	    /* Left arrow key was pressed
	     * this affects the y rotation
	     */
   	    rotLeft = isKeydown;
	    //yspeed -= 0.01f;
	    break;
	case SDLK_F1:
	    /* 'f' key was pressed
	     * this toggles fullscreen mode
	     */
	    if (isKeydown) 
            SDL_WM_ToggleFullScreen( surface );
	    break;
	default:
	    break;
	}

    return;
}

/* general OpenGL initialization function */
bool initalizeGL( GLvoid )
{
            // Enable color tracking	
    glEnable(GL_COLOR_MATERIAL);	
	   	
            // Front material ambient and diffuse colors track glColor	
    glColorMaterial(GL_FRONT,GL_AMBIENT_AND_DIFFUSE);	

    /* Enable smooth shading */
    glShadeModel( GL_SMOOTH );

    /* Set the background black */
    glClearColor( 0.0f, 0.0f, 0.0f, 1.0f );

    /* Depth buffer setup */
    glClearDepth( 1.0f );

    /* Enables Depth Testing */
    glEnable( GL_DEPTH_TEST );

    /* The Type Of Depth Test To Do */
    glDepthFunc( GL_LEQUAL );

    /* Really Nice Perspective Calculations */
    glHint( GL_PERSPECTIVE_CORRECTION_HINT, GL_NICEST );

    /* Setup The Ambient Light */
    glLightfv( GL_LIGHT1, GL_AMBIENT, LightAmbient );

    /* Setup The Diffuse Light */
    glLightfv( GL_LIGHT1, GL_DIFFUSE, LightDiffuse );

    /* Position The Light */
    glLightfv( GL_LIGHT1, GL_POSITION, LightPosition );

    /* Enable Light One */
    glEnable( GL_LIGHT1 );

    /* Full Brightness, 50% Alpha  */
    glColor4f( 1.0f, 1.0f, 1.0f, 1.f);

    /* Blending Function For Translucency Based On Source Alpha Value  */
    glBlendFunc( GL_SRC_ALPHA, GL_ONE );
    
    glEnable( GL_LIGHTING );
    
            // Bright white light  full intensity RGB values	
    //GLfloat ambientLight[] = { 1.0f, 1.0f, 1.0f, 1.0f };	
	    
            // Set light model to use ambient light specified by ambientLight[]	
    //glLightModelfv(GL_LIGHT_MODEL_AMBIENT,ambientLight);	
    
    return( true );
}


/////////////////////////////////////
// KeyState has the state of each key
//
class KeyState
{
    public:
    static bool keyDown[];
};

bool KeyState::keyDown[SDLK_LAST+1];

static KeyState theKeyState;
////////////////////////////////////////
// start here
int main( int argc, char **argv )
{
    /* Flags to pass to SDL_SetVideoMode */
    int videoFlags;
    /* main loop variable */
    int done = FALSE;
    /* used to collect events */
    SDL_Event event;
    /* this holds some info about our display */
    const SDL_VideoInfo *videoInfo;
    /* whether or not the window is active */
    int isActive = TRUE;
    Uint32 tickOfLastDraw = 0;

    /* initialize SDL */
    if ( SDL_Init( SDL_INIT_VIDEO ) < 0 )
	{
	    fprintf( stderr, "Video initialization failed: %s\n",
		     SDL_GetError( ) );
	    Quit( 1 );
	}

    /* Fetch the video info */
    videoInfo = SDL_GetVideoInfo( );

    if ( !videoInfo )
	{
	    fprintf( stderr, "Video query failed: %s\n",
		     SDL_GetError( ) );
	    Quit( 1 );
	}

    /* the flags to pass to SDL_SetVideoMode */
    videoFlags  = SDL_OPENGL;          /* Enable OpenGL in SDL */
    videoFlags |= SDL_GL_DOUBLEBUFFER; /* Enable double buffering */
    videoFlags |= SDL_HWPALETTE;       /* Store the palette in hardware */
    videoFlags |= SDL_RESIZABLE;       /* Enable window resizing */

    /* This checks to see if surfaces can be stored in memory */
    if ( videoInfo->hw_available )
	videoFlags |= SDL_HWSURFACE;
    else
	videoFlags |= SDL_SWSURFACE;

    /* This checks if hardware blits can be done */
    if ( videoInfo->blit_hw )
	videoFlags |= SDL_HWACCEL;

    /* Sets up OpenGL double buffering */
    SDL_GL_SetAttribute( SDL_GL_DOUBLEBUFFER, 1 );

    /* get a SDL surface */
    surface = SDL_SetVideoMode( SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_BPP,
				videoFlags );

    /* Verify there is a surface */
    if ( !surface )
	{
	    fprintf( stderr,  "Video mode set failed: %s\n", SDL_GetError( ) );
	    Quit( 1 );
	}

    /* initialize OpenGL */
    if ( !initalizeGL( ) )
	{
	    fprintf( stderr,  "initGL failed.  Here is the SDL error: %s\n", SDL_GetError( ) );
	    Quit( 1 );
	}

    /* resize the initial window */
    resizeWindow( SCREEN_WIDTH, SCREEN_HEIGHT );

    /* wait for events */
    while ( !done )
	{
	    /* handle the events in the queue */

	    while ( SDL_PollEvent( &event ) )
		{
		    switch( event.type )
			{
			case SDL_ACTIVEEVENT:
			    /* Something's happend with our focus
			     * If we lost focus or we are iconified, we
			     * shouldn't draw the screen
			     */
			    if ( event.active.gain == 0 )
			           isActive = FALSE;
                else
			           isActive = TRUE;
			    break;			    
			case SDL_VIDEORESIZE:
			    /* handle resize event */
			    surface = SDL_SetVideoMode( event.resize.w,
							event.resize.h,
							16, videoFlags );
			    if ( !surface )
				{
				    fprintf( stderr, "Could not get a surface after resize: %s\n", SDL_GetError( ) );
				    Quit( 1 );
				}
			    resizeWindow( event.resize.w, event.resize.h );
			    break;
			case SDL_KEYDOWN:
			case SDL_KEYUP:
			    /* handle key presses */
			    handleKeyPress( &event.key.keysym, event.type == SDL_KEYDOWN );
			    // save the state of the key:
			    theKeyState.keyDown[event.key.keysym.sym] = (event.type == SDL_KEYDOWN);
			    break;
			case SDL_QUIT:
			    /* handle quit requests */
			    done = TRUE;
			    break;
			default:
			    break;
			}
		} //while events queued

        {
            // Wait until next period to draw a frame.
    	    Uint32 timeNow = SDL_GetTicks();
        	if (timeNow - tickOfLastDraw >= 25 )  // 25 ticks/frame = 25 ms/frame = 40 fps
        	{
                tickOfLastDraw = timeNow;
        	    /* draw the scene */
        	    //if ( isActive )
        		drawGLScene( );
        		//countFrameRate();
        	}
    	} 

    } // while !done


    /* clean ourselves up and exit */
    Quit( 0 );

    /* Should never get here */
    return( 0 );
}
//////////////////////
// Calc and log the framerate
// Gather our frames per second 
// Call me when the frame is drawn.
void countFrameRate()
{
    /* These are to calculate our fps */
    static int T0     = 0;
    static int Frames = 0;
    Frames++;
    {
	    int t = SDL_GetTicks();
    	if (t - T0 >= 5000) 
        {
    	    float seconds = (t - T0) / 1000.0f;
    	    float fps = Frames / seconds;
    	    printf("%d frames in %g seconds = %g FPS\n", Frames, seconds, fps);
    	    T0 = t;
    	    Frames = 0;
    	}
    }
}

class thing;
////////////////////////////////////////////////////////////////////
// this is the collection of all objects.
// used for detecting collisions and other interaction.
//
class GlobalMap
{
    public:
    vector<thing *> things;
    vector<thing *> newThings;
    void addThing( thing &theThing )
    {
        newThings.push_back(&theThing);
    }
    // copy from the new list.
    void addAllNew()
    {
        things.insert(things.end(), newThings.begin(), newThings.end());
        newThings.clear();
    }
};

GlobalMap globalMap;


////////////////////////////////////////////////////////////////////
// thing is a 3D sprite.
// takes care of personal position, speed, orientation and spin rate.
//
class thing 
{
    friend class Collider;
    //friend class ColliderBounce;
    public:
    GLfloat x;
    GLfloat y;
    GLfloat z;
    
    GLfloat scaleFactor;
    
    protected:
    GLfloat xRate;
    GLfloat yRate;
    GLfloat zRate;

    GLfloat xRotate;      /* X Rotation */
    GLfloat yRotate;      /* Y Rotation */
    GLfloat zRotate;      /* Y Rotation */
    GLfloat xRotRate;  /*  Rotation Speed */
    GLfloat yRotRate;  /*  Rotation Speed */
    GLfloat zRotRate;  /*  Rotation Speed */
    
    long age;
    float radius;
    //Collider *pCollider;

    thing *parent;  // the stem that spawned me.
    
    bool hitSomething; // set to true when collided.
    private:
        void thingCommonConstruct(GLfloat xi, GLfloat yi, GLfloat zi,  GLint textureIdi);
    public:    
    thing(GLfloat xi, GLfloat yi, GLfloat zi,  GLint textureIdi)
    {
        thingCommonConstruct( xi, yi, zi, textureIdi);
    }
    thing(GLfloat xyz[3])
    {
        thingCommonConstruct(xyz[0], xyz[1], xyz[2], 0);
    }

    void getCenter(float center[3], float &radiusOut)
    {
            center[0] = x; // cube center is (0, 0, 0).
            center[1] = y; 
            center[2] = z; 
            radiusOut = this->radius * scaleFactor;
    }
    
    virtual void think();
    virtual void render()
    {
        glLoadMatrixf(cameraMatrix);  // return to the camera setting.

        
        glTranslatef( x, y, z );
        glRotatef( zRotate, 0.0f, 0.0f, 1.0f); /* Rotate On The Z Axis By ZRotate */
        glRotatef( yRotate, 0.0f, 1.0f, 0.0f); /* Rotate On The Y Axis By yRotate */
        glRotatef( xRotate, 1.0f, 0.0f, 0.0f); /* Rotate On The X Axis By xRotate */

        glScalef( scaleFactor, scaleFactor, scaleFactor  );
        
        drawStem();
        age++;
    }
    
};

// Construct with intial position
void thing::thingCommonConstruct(GLfloat xi, GLfloat yi, GLfloat zi, GLint textureIdi)
{
    this->x = xi;    
    this->y = yi;
    this->z = zi;
    
    this->radius = 1.0f;     // set in initModel() calling calcRadius()

    this->xRate = 0.0f;     // speed of X
    this->yRate = 0.0f;
    this->zRate = 0.0f;

    this->xRotate = 0;      /* X Rotation */
    this->yRotate = 0;      /* Y Rotation */
    this->zRotate = 0;      /* Y Rotation */
    
    this->xRotRate = 0;  /*  Rotation Speed */
    this->yRotRate = 0;  /*  Rotation Speed */
    this->zRotRate = 0;  /*  Rotation Speed */
    
    this->scaleFactor = 1.0f;

    this->age = 0;
    
    //pCollider = 0;
    // register myself with the global map.
    //cerr<< "add new to global map"<<endl;fflush(0);
    
    parent = 0;  // the stem that spawned me.
    hitSomething = false; 
    
    globalMap.addThing(*this);

}

////////////////////////////////////////////////////////////////////
// collider does collision detection and behavior on a thing.
//
class Collider
{
    public:
    static void think(thing &myThing)
    {
        if (isCollision(myThing))
            react(myThing);
    }
    protected:
    /////////////////////////
    // check for collision. returns true if so.
    static bool isCollision(thing &t);

    /////////////////////////////////
    // distance between two points.
    static float distance(float v[3], float w[3])
    {
        return   sqrtf(powf((v[0] - w[0]),2.0f) 
                     + powf((v[1] - w[1]),2.0f) 
                     + powf((v[2] - w[2]),2.0f));
    }
    /////////////////////////
    // called if a collision occurs.
    static void react(thing &t)
    {  
        t.hitSomething = true;
    }
};
///////////////////////////////////////
// Determine if we have collided
bool Collider::isCollision(thing &t)
{  
        float thisCenter[3];
        float thisRadius;
        t.getCenter(thisCenter, thisRadius);
        //        cout << "distance << thisRadius  << radius << pmodel  (from Collider::isCollision)"<< '\n';
        vector<thing *>::iterator it;
        for( it = globalMap.things.begin(); it != globalMap.things.end(); it++ )
        {
            thing &otherThing = **it;
            // skip myself, my parent, and anyone that shares my parent (sybling):
            if (&t == &otherThing 
                        || (t.parent          && t.parent == &otherThing )
                        || (otherThing.parent && t.parent == otherThing.parent )
                        )
                continue;
            // get the other thing's center coordinates and bounding radius.
            float center[3];
            float radius;
            otherThing.getCenter(center, radius);
            //      cout << distance(thisCenter , center) <<" " << thisRadius  <<" " << radius <<" " << otherThing.pModel <<'\n'; 
            
            // collision if centers are closer than there combined radius.
            if (distance(thisCenter , center) <= ( thisRadius  + radius ))
            {
                if (!otherThing.hitSomething)  // only if we noticed first.
                        return true;
            }
        }
        return false;
}

//think about next position in next animation frame
void thing::think()
{

    Collider::think(*this);
    this->x += xRate;    
    this->y += yRate;
    this->z += zRate;

    this->xRotate += xRotRate;      /* X Rotation */
    this->yRotate += yRotRate;      
    this->zRotate += zRotRate;      
}

////////////////////////////////
//  Draw a stem
void drawStem()
{
    // make it skinny:
    glScalef(stemHi/2.0f, stemWi/2.0f,  stemWi/2.0f  );
    //glColor3f( 0.0f, 1.0f, 0.0f);					// green
    //glColor3ub((GLubyte)0,(GLubyte)255,(GLubyte)0);
    /* Start Drawing Quads */
    glBegin( GL_QUADS );
      /* Front Face */
      /* Normal Pointing Towards Viewer */
      glNormal3f( 0.0f, 0.0f, 1.0f );
      /* Point 1 (Front) */
glColor3ub((GLubyte)127,(GLubyte)255,(GLubyte)127);
      glVertex3f( -1.0f, -1.0f,  1.0f );
      /* Point 2 (Front) */
glColor3ub((GLubyte)255,(GLubyte)127,(GLubyte)127);
      glVertex3f(  1.0f, -1.0f,  1.0f );
      /* Point 3 (Front) */
      glVertex3f(  1.0f,  1.0f,  1.0f );
      /* Point 4 (Front) */
glColor3ub((GLubyte)127,(GLubyte)255,(GLubyte)127);
      glVertex3f( -1.0f,  1.0f,  1.0f );

      /* Back Face */
      /* Normal Pointing Away From Viewer */
      glNormal3f( 0.0f, 0.0f, -1.0f);
      /* Point 1 (Back) */
      glVertex3f( -1.0f, -1.0f, -1.0f );
      /* Point 2 (Back) */
      glVertex3f( -1.0f,  1.0f, -1.0f );
      /* Point 3 (Back) */
      glVertex3f(  1.0f,  1.0f, -1.0f );
      /* Point 4 (Back) */
      glVertex3f(  1.0f, -1.0f, -1.0f );

      /* Top Face */
      /* Normal Pointing Up */
      glNormal3f( 0.0f, 1.0f, 0.0f );
      /* Point 1 (Top) */
      glVertex3f( -1.0f,  1.0f, -1.0f );
      /* Point 2 (Top) */
      glVertex3f( -1.0f,  1.0f,  1.0f );
      /* Point 3 (Top) */
      glVertex3f(  1.0f,  1.0f,  1.0f );
      /* Point 4 (Top) */
      glVertex3f(  1.0f,  1.0f, -1.0f );

      /* Bottom Face */
      /* Normal Pointing Down */
      glNormal3f( 0.0f, -1.0f, 0.0f );
      /* Point 1 (Bottom) */
      glVertex3f( -1.0f, -1.0f, -1.0f );
      /* Point 2 (Bottom) */
      glVertex3f(  1.0f, -1.0f, -1.0f );
      /* Point 3 (Bottom) */
      glVertex3f(  1.0f, -1.0f,  1.0f );
      /* Point 4 (Bottom) */
      glVertex3f( -1.0f, -1.0f,  1.0f );

      /* Right face */
      /* Normal Pointing Right */
      glNormal3f( 1.0f, 0.0f, 0.0f);
      /* Point 1 (Right) */
      glVertex3f( 1.0f, -1.0f, -1.0f );
      /* Point 2 (Right) */
      glVertex3f( 1.0f,  1.0f, -1.0f );
      /* Point 3 (Right) */
      glVertex3f( 1.0f,  1.0f,  1.0f );
      /* Point 4 (Right) */
      glVertex3f( 1.0f, -1.0f,  1.0f );

      /* Left Face*/
      /* Normal Pointing Left */
      glNormal3f( -1.0f, 0.0f, 0.0f );
      /* Point 1 (Left) */
      glVertex3f( -1.0f, -1.0f, -1.0f );
      /* Point 2 (Left) */
      glVertex3f( -1.0f, -1.0f,  1.0f );
      /* Point 3 (Left) */
      glVertex3f( -1.0f,  1.0f,  1.0f );
      /* Point 4 (Left) */
      glVertex3f( -1.0f,  1.0f, -1.0f );
    glEnd();

}
void drawGround(float yGround)
{
	glBegin(GL_QUADS);											// Draw A Quad
	{
	    glColor3f( 1.0f, 1.0f, 1.0f  );
		glTexCoord2f(7.0f,4.0f); glVertex3f( 27.0f,yGround,-50.0f);	// Top Right
		glTexCoord2f(0.0f,4.0f); glVertex3f(-27.0f,yGround,-50.0f);	// Top Left
		glTexCoord2f(0.0f,0.0f); glVertex3f(-27.0f,yGround,0.0f);	// Bottom Left
		glTexCoord2f(7.0f,0.0f); glVertex3f( 27.0f,yGround,0.0f);	// Bottom Right
	}
	glEnd();													// Done Drawing Quad

}
////////////////////////////////////////////
// move the camera position and save it in the global matrix.
void positionCamera(bool reset )
{
    glLoadIdentity();
    // Move camera here:
    static float cameraAngle = 0.0f;
    static float cameraZoom  = 0.0f;
    if (reset)
    {
        cameraAngle = 0.0f;
        cameraZoom = 10.0f;
    }
    glTranslatef( 0.0f ,0.0f, -cameraZoom);   // zoom into the center of the rotation.
    glRotatef( cameraAngle, 0.0f, 1.0f, 0.0f); /* Rotate On The Y Axis */
    glTranslatef( 0.0f, 0.0f, 10.0f);
    cameraAngle    += 0.2f;
    cameraZoom     += 0.002f;
    if (cameraAngle  > 360) 
        cameraAngle -= 360;
    
    glGetFloatv(GL_MODELVIEW_MATRIX, cameraMatrix);
}
void positionCamera() {positionCamera(false);}
/////////////////////////
// from polar to cartesion
static float degtorad = 2.0f*M_PI/360.0f;
inline float factorX(float direction[3]){ return cos(degtorad*direction[2])*sin(degtorad*(90+direction[1])); }
inline float factorY(float direction[3]){ return sin(degtorad*direction[2])*sin(degtorad*(90+direction[1])); }
inline float factorZ(float direction[3]){ return cos(degtorad*(90+direction[1])); }


////////////////////////////////////////////
// Thing controlled by arrow keys.
class StemThing : public thing
{
    static const int splitOnGeneration = 3;   // grow this long between splits
    static const int sterileOnGeneration = 9; // stop growing when you are this generation.
    int generation;
    int timeUntilGrow;
    GLfloat direction[3]; 
    
    public:
    StemThing(GLfloat xi, GLfloat yi, GLfloat zi, int generationIn, GLfloat directionIn[3], thing *parentIn) 
        : thing( xi,  yi,  zi, 0 /*=texture*/)
        , generation(generationIn)
    {
        //cerr << "stemThing: construct gen==" << generation << endl;
        //cerr << "pos       x,y,z=   " << xi << ", " << yi << ", " << zi <<endl; 
        //cerr << "direction x,y,z:   " << directionIn[0] << ", "  << directionIn[1] << ", " << directionIn[2]     <<endl; 
        //cerr << "factor    x,y,z:   " << factorX(directionIn) << ", " << factorY(directionIn) << ", " << factorZ(directionIn) << endl; 
        
        parent = parentIn;
        timeUntilGrow = 20 ;//+ (random()&0x7) - 4; // -4 thru 3
        
        // Normalize the angles
        direction[0] = directionIn[0]>360?directionIn[0]-360:directionIn[0];
        direction[1] = directionIn[1]>360?directionIn[1]-360:directionIn[1];
        direction[2] = directionIn[2]>360?directionIn[2]-360:directionIn[2];

        direction[0] = direction[0]<0?direction[0]+360:direction[0];
        direction[1] = direction[1]<0?direction[1]+360:direction[1];
        direction[2] = direction[2]<0?direction[2]+360:direction[2];
        
        // orient the thing in the right direction.
        this->xRotate = direction[0];      /* X Rotation */
        this->yRotate = direction[1];      /* Y Rotation */
        this->zRotate = direction[2];      /* z Rotation */

        // stemWi// allow a little overlap to rule out siblings.
        // use 60% of the hypotenuse of a right triangle as two stems 
        // split at right angles to each other.
        this->radius = (stemHi/4.0f*sqrt(2.0f));  
        //cerr << "stemthing: constructor done." << endl;
    }

    /////////////////////////
    // what to do next frame?
    virtual void think()
    {
        if (generation >= sterileOnGeneration)
            return;
        if (timeUntilGrow < 0)
            return;  // never grow again!
        timeUntilGrow--;
        // Wait this number of frames before growing
        if (timeUntilGrow == 0)
        {        
            thing::think();  // this must be unconditional if need to spin and move.
            //cerr << "Think: grow now." << endl;
            if ( hitSomething && generation > 1)
            {
                GLfloat up[3] = {0.0f
                                , direction[2]==90.0f?180.0f: 0.0f
                                , direction[2]==90.0f?-90.0f:90.0f}; // up facing forward or back
                const GLfloat halfHi = stemHi/2.0f;
                new StemThing(x + halfHi*factorX(direction) + halfHi*factorX(up)
                            , y + halfHi*factorY(direction) + halfHi*factorY(up) //+0.5f
                            , z + halfHi*factorZ(direction) + halfHi*factorZ(up)
                            , 1  // collision baby of a new generation.
                            , up
                            , this);
           
            }
            else if (generation == 0 || 0 != (generation % splitOnGeneration) )
            {   
                // cerr << "Think: grow from head gen==" << generation << endl;
                // grow one on your head
                new StemThing(x+stemHi*factorX(direction)
                            , y+stemHi*factorY(direction)
                            , z+stemHi*factorZ(direction)
                            , generation + 1
                            , direction
                            , this);

                            // zangle is angle around z axis 0-360
                            // yangle is angle around rotated y axis 0-180                                                  
                            // x=r*cos(zangle)*sin(yangle+90)
                            // y=r*sin(zangle)*sin(yangle+90)
                            // z=r*cos(90+yangle)            
            }
            else
            { 
               // cerr << "Think: grow split arms gen==" << generation << endl;
            
                // split right and left like arms:
                // assume that direction[] is a unit vector toward the head, that is x*x+y*y+z*z=1
                GLfloat aoa[3]; // Angle of Arm
                
                GLfloat ax = direction[0];  //[       ]
                GLfloat ay = direction[1];  //[   |_* ]
                GLfloat az = direction[2];  //[   |\  ]
                
                //az+90
                if (ax==0&&ay==0)            {aoa[0]=0.0f;aoa[1]=0.0f;aoa[2]=az+90.0f; } 
                //ax=90 az+90
                else if (ax==0&&ay==90.0f)          {aoa[0]=90.0f;aoa[1]=0.0f;aoa[2]=az+90.0f;} 
                //az-90
                else if (ax==0&&ay==180.0f)         {aoa[0]=0.0f;aoa[1]=180.0f;aoa[2]=az-90.0f;} 
                //az-90
                else if (ax==0&&ay==180.0f)         {aoa[0]=0.0f;aoa[1]=180.0f;aoa[2]=az-90.0f;} 
                //az=270 az+90
                else if (ax==0&&ay==270.0f)            {aoa[0]=270.0f;aoa[1]=0.0f;aoa[2]=az+90.0f;} 
                //ax=0 ay=270 az+90
                else if (ax==90.0f&&ay==0)          {aoa[0]=0.0f;aoa[1]=270.0f;aoa[2]=az+90.0f;} 
                else if (ax==90.0f&&ay==90.0)       {aoa[0]=90.0f;aoa[1]=0.0f;aoa[2]=az;} //test me
                else if (ax==90.0f&&ay==180.0f)     {aoa[0]=90.0f;aoa[1]=270.0f;aoa[2]=az;} //test me
                else if (ax==90.0f&&ay==270.0f)     {aoa[0]=90.0f;aoa[1]=0.0f;aoa[2]=az+180.0f;} //test me
                // skip for now if (ax==180 
                else if (ax==270.0f&&ay==180.0f)    {aoa[0]=0.0f;aoa[1]=270.0f;aoa[2]=az-90.0f;} //testme
                // ay=90 az+90
                else if (ax==270.0f&&ay==0)         {aoa[0]=0.0f;aoa[1]=90.0f;aoa[2]=az+90.0f;} //or 270,90,az  //tm
                else if (ax==270.0f&&ay==90.0f)     {aoa[0]=270.0f;aoa[1]=0.0f;aoa[2]=az+180.0f;} //tm
                else if (ax==270.0f&&ay==180.0f)    {aoa[0]=270.0f;aoa[1]=270.0f;aoa[2]=az;} 
                else if (ax==270.0f&&ay==270.0f)    {aoa[0]=270.0f;aoa[1]=0.0f;aoa[2]=az;} //test me
                // done?
                
                else 
                {    
                    //cerr<< "Was bad left, no rule matches!! look at was below."            << endl; 

                    aoa[0] =  (ay==270)? ax-90: (ay==90) ? ax+90: ax;
                    aoa[1] =  (ay==270)?  0   : (ay==90) ? 0    : ay;
                    aoa[2] =  az+90;
                    //aoa[0] =  ax+45.0f;
                    //aoa[1] =  ay+45.0f;
                    //aoa[2] =  az+45.0f;
                }
                
//cerr << "was   " << direction[0]     << ", "  << direction[1]     << ", " << direction[2]     <<endl; 
//cerr << "left  " << aoa[0] << ", "  << aoa[1] << ", " << aoa[2] <<endl; 
    
        
                const GLfloat halfHi = stemHi/2.0f;
    
                
                new StemThing(x + halfHi*factorX(direction) + halfHi*factorX(aoa)
                            , y + halfHi*factorY(direction) + halfHi*factorY(aoa) //+0.5f
                            , z + halfHi*factorZ(direction) + halfHi*factorZ(aoa)
                            , generation+1 /*=generation*/
                            , aoa
                            , this);

                // Now the other arm
                //aoa[0] =  ( direction[0]);    //[   |_   ]
                //aoa[1] =  ( direction[1]-90); //[   |\   ] //note that backslashes extend coments! 
                //aoa[2] =  ( direction[2]);    //[     *  ]
                {                
                    aoa[0] =  ax;
                    aoa[1] =  ay-90;
                    aoa[2] =  az;
                }
    
                //cerr << "front "<< aoa[0] << ", "  << aoa[1] << ", " << aoa[2] <<endl; 
    
                new StemThing(x + halfHi*factorX(direction) + halfHi*factorX(aoa)
                            , y + halfHi*factorY(direction) + halfHi*factorY(aoa) //+0.5f
                            , z + halfHi*factorZ(direction) + halfHi*factorZ(aoa)
                            , generation+1 /*=generation*/
                            , aoa
                            , this);
             }
        }           
    }
    //    virtual void render() {thing::render();};
};

//////////////////
// Random numbers are only used to set the initial conditions.
// low number, high number.
// get a sequence of random integers within a range [low, high]. 
// returns an integer converted to a float.
float randrange(int low, int high)
{    
    return (float)(low + rand()%(high - low + 1));
}
    
/////////////////////////////////
// Here goes our drawing code 
int drawGLScene( GLvoid )
{
//cout    << "cos 90=0=" << cos(3.14159f/2.0f) 
//<< endl << "sin 90=1=" << sin(3.14159f/2.0f) <<endl;

    float const static yGround = -3.0f;
    /* Clear The Screen And The Depth Buffer */
    glClear( GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT );
    static bool isInitialized = false;
    if (!isInitialized)
    {
        isInitialized = true;
        positionCamera(true); // true means reset.
        while ( globalMap.things.size() > 0 ) 
        {
            thing *delThing = globalMap.things.back();
            globalMap.things.erase( --(globalMap.things.end()) );
            delete delThing;
        }
        srand(time(NULL));
        // initialize logical stuff here 
        GLfloat atGround = yGround+stemHi/2.0f;
        GLfloat up1[3] = { 0.0f,   0.0f,  90.0f }; // right facing forward
        GLfloat up2[3] = { 0.0f, 180.0f, -90.0f }; // right facing backward

        new StemThing(stemHi*randrange(-10,10), atGround
                        , -10.0f+stemHi*randrange(-4,4), 1, randrange(0,1)?up1:up2, 0);
        new StemThing(stemHi*randrange(-10,10), atGround
                        , -10.0f+stemHi*randrange(-4,4), 1, randrange(0,1)?up1:up2, 0);

        if ( randrange(0,1) ) 
        {
                GLfloat up[3] = { 0.0f, 0.0f, 90.0f }; // right facing forward
                new StemThing(stemHi*randrange(-10,10), atGround
                                , -10.0f+stemHi*randrange(-4,4), 1, randrange(0,1)?up1:up2, 0);
        }
        if ( randrange(0,1) ) 
        {
                GLfloat up[3] = { 0.0f, 0.0f, 90.0f }; // right facing forward
                new StemThing(stemHi*randrange(-10,10), atGround
                                , -10.0f+stemHi*randrange(-4,4), 1, randrange(0,1)?up1:up2, 0);
        }
        //GLfloat sp[3] = {0.0f, -90.0f, 0.0f}; // forward facing left
        //new StemThing(0.0f, yGround+stemHi/2.0f, -10.0f, 1, sp, 0);
        //GLfloat to[3] = {0.0f, 90.0f, 90.0f}; // back away facing up
        //new StemThing(2.0f, yGround+stemHi/2.0f, -10.0f, 1, to, 0);
        //GLfloat al[3] = {0.0f, 180.0f, 90.0f}; // down facing back
        //new StemThing(5.0f, yGround+stemHi/2.0f, -10.0f, 1, al, 0);
        GLfloat lt[3] = {0.0f, 180.0f, -90.0f}; // up facing forward
        new StemThing(-0.2f, yGround+stemHi/2.0f, -10.0f, 1, lt, 0);
    }
    
    // move the camera by setting the camera matrix:
    positionCamera();

    // Draw the ground as a square:
    //drawGround(yGround);

        
            //cerr << "drawGLScene: Begin thinking loop." << endl;
    // Make all things think:
            //cerr<< "before think " << globalMap.things.size()<<endl;  
    vector<thing*>::iterator it;
            //static int cycle =60;if (cycle){cycle--;
    for( it = globalMap.things.begin(); it != globalMap.things.end(); it++ )
        (*it)->think();
            //}       //cerr << "drawGLScene: End thinking loop." << endl;
    // add things created in that last think loop. This is so that new inserts don't cause problems.
            //cerr<< "after think " << globalMap.things.size()<<endl;       
    globalMap.addAllNew();    
            //cerr<< "after addallnew " << globalMap.things.size()<<endl;       
            //cerr << "drawGLScene: Begin render loop." << endl;
    // Make all things show:
    for( it = globalMap.things.begin(); it != globalMap.things.end(); it++ )
        (*it)->render();
            //cerr<< "after render "<< globalMap.things.size()<<endl;
            
    // Determine if this is the end and we need to restart.        
    const static int framesToShowAfterEnd = 100;
    static int framesCountDownToRestart = framesToShowAfterEnd;
    static int lastSize = 0;
    int newSize = globalMap.things.size();            
    if ((lastSize==newSize || newSize>500) )
    {
        if (--framesCountDownToRestart == 0)
                 isInitialized = false;  // restart now!
    }
    else
    {
        // Still adding more stems, reset the countdown.
        lastSize = newSize;
        framesCountDownToRestart = framesToShowAfterEnd;
    }
    /* Draw it to the screen */
    SDL_GL_SwapBuffers();
    
    return( TRUE );
}


