#ifndef CNEURALNET_H
#define CNEURALNET_H
//------------------------------------------------------------------------
//
//	Name: CNeuralNet.h
//
//  Author: Mat Buckland 2002
//
//  Desc: Class for creating a feedforward neural net.
//-------------------------------------------------------------------------
#include <vector>
#include <fstream>
#include <math.h>


#include "utils.h"
#include "CParams.h"


using namespace std;


const double ActivationResponse = 1.0;


//-------------------------------Sigmoid function-------------------------
//
//------------------------------------------------------------------------
inline double Sigmoid(const double netinput, const double response)
{
	return ( 1 / ( 1 + exp(-netinput / response)));
}


//------------------------------------------------------------------------
//
//  The following code creates a lookup table to speed up the sigmoid 
//  function
//------------------------------------------------------------------------
const double SigmoidRange = 20.0;
const int    NumSamples   = 4000;
const double SampleSlice  = SigmoidRange*2/NumSamples;

static double SigmoidTable[NumSamples];


static bool CreateLookupTable()
{
  int idx = 0;

  for (double d=-SigmoidRange; d<SigmoidRange; d+=SampleSlice)
  {
    SigmoidTable[idx++] = Sigmoid(d, ActivationResponse);
  }

  return true;
}

static bool Done = CreateLookupTable();

inline double FastSigmoid(const double val)
{
  if (val < -SigmoidRange)
  {
    return 0;
  }

  else if (val > SigmoidRange)
  {
    return 1.0;
  }

  else
  {
    //calculate the index into the sigmoid table
    int idx = (int)((val + SigmoidRange)/SampleSlice);

    return SigmoidTable[idx];
  }
}


//-------------------------------------------------------------------
//	define neuron struct
//-------------------------------------------------------------------
struct SNeuron
{
	//the number of inputs into the neuron
	int				      m_NumInputs;

	//the weights for each input
	vector<double>	m_vecWeight;


	//ctor
	SNeuron(int NumInputs);
};


//---------------------------------------------------------------------
//	struct to hold a layer of neurons.
//---------------------------------------------------------------------

struct SNeuronLayer
{
	//the number of neurons in this layer
	int					      m_NumNeurons;

	//the layer of neurons
	vector<SNeuron>		m_vecNeurons;

	SNeuronLayer(int NumNeurons, 
				       int NumInputsPerNeuron);
};


//----------------------------------------------------------------------
//	neural net class
//----------------------------------------------------------------------

class CNeuralNet
{
	
private:
	
	int					m_NumInputs;

	int					m_NumOutputs;

	int					m_NumHiddenLayers;

	int					m_NeuronsPerHiddenLyr;

	//storage for each layer of neurons including the output layer
	vector<SNeuronLayer>	m_vecLayers;
 

  //vectors used in the update function. When this class is instantiated
  //these vectors are allocated enough memory to store the max amount
  //of elements (m_iMaxBufferSize) required by the update function. This 
  //is simply the larger of the amounts m_NumInputs, m_NumOutputs or
  //m_NeuronsPerHiddenLyr
  vector<double> m_Inputs,
                 m_Outputs;

  int CalculateMaxBufferSize();

public:

	CNeuralNet();

	void			      CreateNet();

	//gets the weights from the NN
	vector<double>	GetWeights()const;

	//returns total number of weights in net
	int				      GetNumberOfWeights()const;

	//replaces the weights with new ones
	void			      PutWeights(const vector<double> &weights);

	//calculates the outputs from a set of inputs
	vector<double>	Update(const vector<double> &inputs);

	//sigmoid response curve
	inline double	  Sigmoid(const double activation, const double response);

  vector<int>     CalculateSplitPoints() const;

};
				



#endif